/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.mr.hive;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.function.BinaryOperator;
import java.util.function.Function;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hive.metastore.api.ColumnStatisticsObj;
import org.apache.hadoop.hive.ql.Context;
import org.apache.hadoop.hive.ql.io.sarg.SearchArgument;
import org.apache.hadoop.hive.ql.metadata.HiveUtils;
import org.apache.hadoop.hive.ql.parse.AlterTableExecuteSpec;
import org.apache.hadoop.hive.ql.plan.PlanUtils;
import org.apache.hadoop.hive.ql.session.SessionStateUtil;
import org.apache.iceberg.DeleteFiles;
import org.apache.iceberg.ManageSnapshots;
import org.apache.iceberg.PartitionData;
import org.apache.iceberg.PartitionSpec;
import org.apache.iceberg.RowLevelOperationMode;
import org.apache.iceberg.Schema;
import org.apache.iceberg.SnapshotRef;
import org.apache.iceberg.StructLike;
import org.apache.iceberg.Table;
import org.apache.iceberg.TableProperties;
import org.apache.iceberg.UpdatePartitionSpec;
import org.apache.iceberg.catalog.TableIdentifier;
import org.apache.iceberg.expressions.Expression;
import org.apache.iceberg.expressions.Expressions;
import org.apache.iceberg.mr.Catalogs;
import org.apache.iceberg.mr.hive.HiveIcebergFilterFactory;
import org.apache.iceberg.types.Types;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IcebergTableUtil {
    private static final Logger LOG = LoggerFactory.getLogger(IcebergTableUtil.class);

    private IcebergTableUtil() {
    }

    static Table getTable(Configuration configuration, org.apache.hadoop.hive.metastore.api.Table hmsTable, boolean skipCache) {
        Properties properties = new Properties();
        properties.setProperty("name", TableIdentifier.of(hmsTable.getDbName(), hmsTable.getTableName()).toString());
        properties.setProperty("location", hmsTable.getSd().getLocation());
        hmsTable.getParameters().computeIfPresent("iceberg.catalog", (k, v) -> {
            properties.setProperty((String)k, (String)v);
            return v;
        });
        return IcebergTableUtil.getTable(configuration, properties, skipCache);
    }

    static Table getTable(Configuration configuration, org.apache.hadoop.hive.metastore.api.Table hmsTable) {
        return IcebergTableUtil.getTable(configuration, hmsTable, false);
    }

    static Table getTable(Configuration configuration, Properties properties, boolean skipCache) {
        String metaTable = properties.getProperty("metaTable");
        String tableName = properties.getProperty("name");
        String location = properties.getProperty("location");
        if (metaTable != null) {
            properties.setProperty("name", tableName + "." + metaTable);
            properties.setProperty("location", location + "#" + metaTable);
        }
        String tableIdentifier = properties.getProperty("name");
        Function<Void, Table> tableLoadFunc = unused -> {
            Table tab = Catalogs.loadTable(configuration, properties);
            SessionStateUtil.addResource((Configuration)configuration, (String)tableIdentifier, (Object)tab);
            return tab;
        };
        if (skipCache) {
            return tableLoadFunc.apply(null);
        }
        return SessionStateUtil.getResource((Configuration)configuration, (String)tableIdentifier).filter(o -> o instanceof Table).map(o -> (Table)o).orElseGet(() -> {
            LOG.debug("Iceberg table {} is not found in QueryState. Loading table from configured catalog", (Object)tableIdentifier);
            return (Table)tableLoadFunc.apply(null);
        });
    }

    static Table getTable(Configuration configuration, Properties properties) {
        return IcebergTableUtil.getTable(configuration, properties, false);
    }

    static Optional<Path> getColStatsPath(Table table) {
        return IcebergTableUtil.getColStatsPath(table, table.currentSnapshot().snapshotId());
    }

    static Optional<Path> getColStatsPath(Table table, long snapshotId) {
        return table.statisticsFiles().stream().filter(stats -> stats.snapshotId() == snapshotId).filter(stats -> stats.blobMetadata().stream().anyMatch(metadata -> ColumnStatisticsObj.class.getSimpleName().equals(metadata.type()))).map(stats -> new Path(stats.path())).findAny();
    }

    public static PartitionSpec spec(Configuration configuration, Schema schema) {
        List partitionTransformSpecList = SessionStateUtil.getResource((Configuration)configuration, (String)"partition_transform_spec").map(o -> (List)o).orElseGet(() -> null);
        if (partitionTransformSpecList == null) {
            LOG.debug("Iceberg partition transform spec is not found in QueryState.");
            return null;
        }
        PartitionSpec.Builder builder = PartitionSpec.builderFor(schema);
        partitionTransformSpecList.forEach(spec -> {
            switch (spec.getTransformType()) {
                case IDENTITY: {
                    builder.identity(spec.getColumnName().toLowerCase());
                    break;
                }
                case YEAR: {
                    builder.year(spec.getColumnName());
                    break;
                }
                case MONTH: {
                    builder.month(spec.getColumnName());
                    break;
                }
                case DAY: {
                    builder.day(spec.getColumnName());
                    break;
                }
                case HOUR: {
                    builder.hour(spec.getColumnName());
                    break;
                }
                case TRUNCATE: {
                    builder.truncate(spec.getColumnName(), (Integer)spec.getTransformParam().get());
                    break;
                }
                case BUCKET: {
                    builder.bucket(spec.getColumnName(), (Integer)spec.getTransformParam().get());
                }
            }
        });
        return builder.build();
    }

    public static void updateSpec(Configuration configuration, Table table) {
        PartitionSpec newPartitionSpec = IcebergTableUtil.spec(configuration, table.schema());
        if (newPartitionSpec == null) {
            LOG.debug("Iceberg Partition spec is not updated due to empty partition spec definition.");
            return;
        }
        UpdatePartitionSpec updatePartitionSpec = table.updateSpec().caseSensitive(false);
        table.spec().fields().forEach(field -> updatePartitionSpec.removeField(field.name()));
        List partitionTransformSpecList = SessionStateUtil.getResource((Configuration)configuration, (String)"partition_transform_spec").map(o -> (List)o).orElseGet(() -> null);
        partitionTransformSpecList.forEach(spec -> {
            switch (spec.getTransformType()) {
                case IDENTITY: {
                    updatePartitionSpec.addField(spec.getColumnName());
                    break;
                }
                case YEAR: {
                    updatePartitionSpec.addField(Expressions.year(spec.getColumnName()));
                    break;
                }
                case MONTH: {
                    updatePartitionSpec.addField(Expressions.month(spec.getColumnName()));
                    break;
                }
                case DAY: {
                    updatePartitionSpec.addField(Expressions.day(spec.getColumnName()));
                    break;
                }
                case HOUR: {
                    updatePartitionSpec.addField(Expressions.hour(spec.getColumnName()));
                    break;
                }
                case TRUNCATE: {
                    updatePartitionSpec.addField(Expressions.truncate(spec.getColumnName(), (Integer)spec.getTransformParam().get()));
                    break;
                }
                case BUCKET: {
                    updatePartitionSpec.addField(Expressions.bucket(spec.getColumnName(), (Integer)spec.getTransformParam().get()));
                }
            }
        });
        updatePartitionSpec.commit();
    }

    public static boolean isBucketed(Table table) {
        return table.spec().fields().stream().anyMatch(f -> f.transform().toString().startsWith("bucket["));
    }

    public static void rollback(Table table, AlterTableExecuteSpec.RollbackSpec.RollbackType type, Long value) {
        ManageSnapshots manageSnapshots = table.manageSnapshots();
        if (type == AlterTableExecuteSpec.RollbackSpec.RollbackType.TIME) {
            LOG.debug("Trying to rollback iceberg table to snapshot before timestamp {}", (Object)value);
            manageSnapshots.rollbackToTime(value);
        } else {
            LOG.debug("Trying to rollback iceberg table to snapshot ID {}", (Object)value);
            manageSnapshots.rollbackTo(value);
        }
        manageSnapshots.commit();
    }

    public static void setCurrentSnapshot(Table table, String value) {
        long snapshotId;
        ManageSnapshots manageSnapshots = table.manageSnapshots();
        try {
            snapshotId = Long.parseLong(value);
            LOG.debug("Rolling the iceberg table {} from snapshot id {} to snapshot ID {}", new Object[]{table.name(), table.currentSnapshot().snapshotId(), snapshotId});
        }
        catch (NumberFormatException e) {
            String refName = PlanUtils.stripQuotes((String)value);
            snapshotId = Optional.ofNullable(table.refs().get(refName)).map(SnapshotRef::snapshotId).orElseThrow(() -> new IllegalArgumentException(String.format("SnapshotRef %s does not exist", refName)));
            LOG.debug("Rolling the iceberg table {} from snapshot id {} to the snapshot ID {} of SnapshotRef {}", new Object[]{table.name(), table.currentSnapshot().snapshotId(), snapshotId, refName});
        }
        manageSnapshots.setCurrentSnapshot(snapshotId);
        manageSnapshots.commit();
    }

    public static void fastForwardBranch(Table table, String sourceBranch, String targetBranch) {
        LOG.debug("Fast Forwarding the iceberg table {} branch {} to {}", new Object[]{table.name(), sourceBranch, targetBranch});
        table.manageSnapshots().fastForwardBranch(sourceBranch, targetBranch).commit();
    }

    public static void cherryPick(Table table, long snapshotId) {
        LOG.debug("Cherry-Picking {} to {}", (Object)snapshotId, (Object)table.name());
        table.manageSnapshots().cherrypick(snapshotId).commit();
    }

    public static boolean isV2Table(Map<String, String> props) {
        return props != null && "2".equals(props.get("format-version"));
    }

    public static boolean isCopyOnWriteMode(Context.Operation operation, BinaryOperator<String> props) {
        String mode = null;
        switch (operation) {
            case DELETE: {
                mode = (String)props.apply("write.delete.mode", TableProperties.DELETE_MODE_DEFAULT);
                break;
            }
            case UPDATE: {
                mode = (String)props.apply("write.update.mode", TableProperties.UPDATE_MODE_DEFAULT);
                break;
            }
            case MERGE: {
                mode = (String)props.apply("write.merge.mode", TableProperties.MERGE_MODE_DEFAULT);
            }
        }
        return RowLevelOperationMode.COPY_ON_WRITE.modeName().equalsIgnoreCase(mode);
    }

    public static void performMetadataDelete(Table icebergTable, String branchName, SearchArgument sarg) {
        Expression exp = HiveIcebergFilterFactory.generateFilterExpression(sarg);
        DeleteFiles deleteFiles = icebergTable.newDelete();
        if (StringUtils.isNotEmpty((CharSequence)branchName)) {
            deleteFiles = (DeleteFiles)deleteFiles.toBranch(HiveUtils.getTableSnapshotRef((String)branchName));
        }
        deleteFiles.deleteFromRowFilter(exp).commit();
    }

    public static PartitionData toPartitionData(StructLike key, Types.StructType keyType) {
        PartitionData data = new PartitionData(keyType);
        for (int i = 0; i < keyType.fields().size(); ++i) {
            Object val = key.get(i, keyType.fields().get(i).type().typeId().javaClass());
            if (val == null) continue;
            data.set(i, val);
        }
        return data;
    }
}

