/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.actions;

import java.io.Closeable;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.iceberg.Table;
import org.apache.iceberg.relocated.com.google.common.annotations.VisibleForTesting;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.collect.Lists;
import org.apache.iceberg.relocated.com.google.common.collect.Queues;
import org.apache.iceberg.relocated.com.google.common.collect.Sets;
import org.apache.iceberg.relocated.com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

abstract class BaseCommitService<T>
implements Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(BaseCommitService.class);
    public static final long TIMEOUT_IN_MS_DEFAULT = TimeUnit.MINUTES.toMillis(120L);
    private final Table table;
    private final ExecutorService committerService;
    private final ConcurrentLinkedQueue<T> completedRewrites;
    private final ConcurrentLinkedQueue<String> inProgressCommits;
    private final ConcurrentLinkedQueue<T> committedRewrites;
    private final int rewritesPerCommit;
    private final AtomicBoolean running = new AtomicBoolean(false);
    private final long timeoutInMS;

    BaseCommitService(Table table, int rewritesPerCommit) {
        this(table, rewritesPerCommit, TIMEOUT_IN_MS_DEFAULT);
    }

    BaseCommitService(Table table, int rewritesPerCommit, long timeoutInMS) {
        this.table = table;
        LOG.info("Creating commit service for table {} with {} groups per commit", (Object)table, (Object)rewritesPerCommit);
        this.rewritesPerCommit = rewritesPerCommit;
        this.timeoutInMS = timeoutInMS;
        this.committerService = Executors.newSingleThreadExecutor(new ThreadFactoryBuilder().setNameFormat("Committer-Service").build());
        this.completedRewrites = Queues.newConcurrentLinkedQueue();
        this.committedRewrites = Queues.newConcurrentLinkedQueue();
        this.inProgressCommits = Queues.newConcurrentLinkedQueue();
    }

    protected abstract void commitOrClean(Set<T> var1);

    protected abstract void abortFileGroup(T var1);

    public void start() {
        Preconditions.checkState(this.running.compareAndSet(false, true), "Commit service already started");
        LOG.info("Starting commit service for {}", (Object)this.table);
        this.committerService.execute(() -> {
            while (this.running.get() || this.completedRewrites.size() > 0 || this.inProgressCommits.size() > 0) {
                try {
                    if (this.completedRewrites.size() == 0 && this.inProgressCommits.size() == 0) {
                        Thread.sleep(100L);
                    }
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    throw new RuntimeException("Interrupted while processing commits", e);
                }
                if (this.running.get() || this.completedRewrites.size() <= 0) continue;
                this.commitReadyCommitGroups();
            }
        });
    }

    public void offer(T group) {
        LOG.debug("Offered to commit service: {}", group);
        Preconditions.checkState(this.running.get(), "Cannot add rewrites to a service which has already been closed");
        this.completedRewrites.add(group);
        this.commitReadyCommitGroups();
    }

    public List<T> results() {
        Preconditions.checkState(this.committerService.isShutdown(), "Cannot get results from a service which has not been closed");
        return Lists.newArrayList(this.committedRewrites.iterator());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() {
        Preconditions.checkState(this.running.compareAndSet(true, false), "Cannot close already closed commit service");
        LOG.info("Closing commit service for {} waiting for all commits to finish", (Object)this.table);
        this.committerService.shutdown();
        boolean timeout = false;
        try {
            if (!this.committerService.awaitTermination(this.timeoutInMS, TimeUnit.MILLISECONDS)) {
                LOG.warn("Commit operation did not complete within {} minutes ({} ms) of the all files being rewritten. This may mean that some changes were not successfully committed to the table.", (Object)TimeUnit.MILLISECONDS.toMinutes(this.timeoutInMS), (Object)this.timeoutInMS);
                timeout = true;
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new RuntimeException("Cannot complete commit for rewrite, commit service interrupted", e);
        }
        if (!this.completedRewrites.isEmpty() && timeout) {
            LOG.error("Attempting to cleanup uncommitted file groups");
            ConcurrentLinkedQueue<T> concurrentLinkedQueue = this.completedRewrites;
            synchronized (concurrentLinkedQueue) {
                while (!this.completedRewrites.isEmpty()) {
                    this.abortFileGroup(this.completedRewrites.poll());
                }
            }
        }
        Preconditions.checkArgument(!timeout && this.completedRewrites.isEmpty(), "Timeout occurred when waiting for commits to complete. {} file groups committed. {} file groups remain uncommitted. Retry this operation to attempt rewriting the failed groups.", this.committedRewrites.size(), this.completedRewrites.size());
        Preconditions.checkState(this.completedRewrites.isEmpty(), "File groups offered after service was closed, they were not successfully committed.");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void commitReadyCommitGroups() {
        HashSet<T> batch = null;
        if (this.canCreateCommitGroup()) {
            ConcurrentLinkedQueue<T> concurrentLinkedQueue = this.completedRewrites;
            synchronized (concurrentLinkedQueue) {
                if (this.canCreateCommitGroup()) {
                    batch = Sets.newHashSetWithExpectedSize(this.rewritesPerCommit);
                    for (int i = 0; i < this.rewritesPerCommit && !this.completedRewrites.isEmpty(); ++i) {
                        batch.add(this.completedRewrites.poll());
                    }
                }
            }
        }
        if (batch != null) {
            String inProgressCommitToken = UUID.randomUUID().toString();
            this.inProgressCommits.add(inProgressCommitToken);
            try {
                this.commitOrClean(batch);
                this.committedRewrites.addAll(batch);
            }
            catch (Exception e) {
                LOG.error("Failure during rewrite commit process, partial progress enabled. Ignoring", (Throwable)e);
            }
            this.inProgressCommits.remove(inProgressCommitToken);
        }
    }

    @VisibleForTesting
    boolean canCreateCommitGroup() {
        boolean fullCommitGroup = this.completedRewrites.size() >= this.rewritesPerCommit;
        boolean writingComplete = !this.running.get() && this.completedRewrites.size() > 0;
        return fullCommitGroup || writingComplete;
    }

    @VisibleForTesting
    boolean completedRewritesAllCommitted() {
        return this.completedRewrites.isEmpty() && this.inProgressCommits.isEmpty();
    }
}

