/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hive.metastore;

import com.google.common.base.Preconditions;
import java.io.IOException;
import java.security.PrivilegedExceptionAction;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.Optional;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hive.metastore.auth.HttpAuthenticationException;
import org.apache.hadoop.hive.metastore.auth.jwt.JWTValidator;
import org.apache.hadoop.hive.metastore.conf.MetastoreConf;
import org.apache.hadoop.security.SecurityUtil;
import org.apache.hadoop.security.UserGroupInformation;
import org.eclipse.jetty.util.ssl.SslContextFactory;
import org.pac4j.core.context.JEEContext;
import org.pac4j.core.context.WebContext;
import org.pac4j.core.credentials.TokenCredentials;
import org.pac4j.core.credentials.extractor.BearerAuthExtractor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ServletSecurity {
    private static final Logger LOG = LoggerFactory.getLogger(ServletSecurity.class);
    static final String X_USER = "x-actor-username";
    private final boolean isSecurityEnabled;
    private final boolean jwtAuthEnabled;
    private JWTValidator jwtValidator = null;
    private final Configuration conf;

    ServletSecurity(Configuration conf, boolean jwt) {
        this.conf = conf;
        this.isSecurityEnabled = UserGroupInformation.isSecurityEnabled();
        this.jwtAuthEnabled = jwt;
    }

    public void init() throws ServletException {
        if (this.jwtAuthEnabled) {
            try {
                this.jwtValidator = new JWTValidator(this.conf);
            }
            catch (Exception e) {
                throw new ServletException("Failed to initialize ServletSecurity. Error: " + e);
            }
        }
    }

    public void execute(HttpServletRequest request, HttpServletResponse response, MethodExecutor executor) throws ServletException, IOException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Logging headers in " + request.getMethod() + " request");
            Enumeration headerNames = request.getHeaderNames();
            while (headerNames.hasMoreElements()) {
                String headerName = (String)headerNames.nextElement();
                LOG.debug("Header: [{}], Value: [{}]", (Object)headerName, (Object)request.getHeader(headerName));
            }
        }
        try {
            UserGroupInformation clientUgi;
            String userFromHeader = this.extractUserName(request, response);
            if (this.isSecurityEnabled || this.jwtAuthEnabled) {
                LOG.info("Creating proxy user for: {}", (Object)userFromHeader);
                clientUgi = UserGroupInformation.createProxyUser((String)userFromHeader, (UserGroupInformation)UserGroupInformation.getLoginUser());
            } else {
                LOG.info("Creating remote user for: {}", (Object)userFromHeader);
                clientUgi = UserGroupInformation.createRemoteUser((String)userFromHeader);
            }
            PrivilegedExceptionAction<Void> action = () -> {
                executor.execute(request, response);
                return null;
            };
            try {
                clientUgi.doAs(action);
            }
            catch (InterruptedException e) {
                LOG.error("Exception when executing http request as user: " + clientUgi.getUserName(), (Throwable)e);
                Thread.currentThread().interrupt();
            }
            catch (RuntimeException e) {
                LOG.error("Exception when executing http request as user: " + clientUgi.getUserName(), (Throwable)e);
                throw new ServletException((Throwable)e);
            }
        }
        catch (HttpAuthenticationException e) {
            response.setStatus(401);
            response.getWriter().println("Authentication error: " + e.getMessage());
            LOG.error("Authentication error: ", (Throwable)e);
        }
    }

    private String extractUserName(HttpServletRequest request, HttpServletResponse response) throws HttpAuthenticationException {
        String user;
        if (!this.jwtAuthEnabled) {
            String userFromHeader = request.getHeader(X_USER);
            if (userFromHeader == null || userFromHeader.isEmpty()) {
                throw new HttpAuthenticationException("User header x-actor-username missing in request");
            }
            return userFromHeader;
        }
        String signedJwt = this.extractBearerToken(request, response);
        if (signedJwt == null) {
            throw new HttpAuthenticationException("Couldn't find bearer token in the auth header in the request");
        }
        try {
            user = this.jwtValidator.validateJWTAndExtractUser(signedJwt);
            Preconditions.checkNotNull((Object)user, (Object)"JWT needs to contain the user name as subject");
            Preconditions.checkState((!user.isEmpty() ? 1 : 0) != 0, (Object)"User name should not be empty in JWT");
            LOG.info("Successfully validated and extracted user name {} from JWT in Auth header in the request", (Object)user);
        }
        catch (Exception e) {
            throw new HttpAuthenticationException("Failed to validate JWT from Bearer token in Authentication header", e);
        }
        return user;
    }

    private String extractBearerToken(HttpServletRequest request, HttpServletResponse response) {
        BearerAuthExtractor extractor = new BearerAuthExtractor();
        Optional tokenCredentials = extractor.extract((WebContext)new JEEContext(request, response));
        return tokenCredentials.map(TokenCredentials::getToken).orElse(null);
    }

    static void loginServerPincipal(Configuration conf) throws IOException {
        LOG.info(" Checking if security is enabled");
        if (UserGroupInformation.isSecurityEnabled()) {
            LOG.info("Logging in via keytab while starting HTTP metastore");
            String kerberosName = SecurityUtil.getServerPrincipal((String)MetastoreConf.getVar((Configuration)conf, (MetastoreConf.ConfVars)MetastoreConf.ConfVars.KERBEROS_PRINCIPAL), (String)"0.0.0.0");
            String keyTabFile = MetastoreConf.getVar((Configuration)conf, (MetastoreConf.ConfVars)MetastoreConf.ConfVars.KERBEROS_KEYTAB_FILE);
            UserGroupInformation.loginUserFromKeytab((String)kerberosName, (String)keyTabFile);
        } else {
            LOG.info("Security is not enabled. Not logging in via keytab");
        }
    }

    static SslContextFactory createSslContextFactory(Configuration conf) throws IOException {
        boolean useSsl = MetastoreConf.getBoolVar((Configuration)conf, (MetastoreConf.ConfVars)MetastoreConf.ConfVars.USE_SSL);
        if (!useSsl) {
            return null;
        }
        String keyStorePath = MetastoreConf.getVar((Configuration)conf, (MetastoreConf.ConfVars)MetastoreConf.ConfVars.SSL_KEYSTORE_PATH).trim();
        if (keyStorePath.isEmpty()) {
            throw new IllegalArgumentException(MetastoreConf.ConfVars.SSL_KEYSTORE_PATH.toString() + " Not configured for SSL connection");
        }
        String keyStorePassword = MetastoreConf.getPassword((Configuration)conf, (MetastoreConf.ConfVars)MetastoreConf.ConfVars.SSL_KEYSTORE_PASSWORD);
        String keyStoreType = MetastoreConf.getVar((Configuration)conf, (MetastoreConf.ConfVars)MetastoreConf.ConfVars.SSL_KEYSTORE_TYPE).trim();
        String keyStoreAlgorithm = MetastoreConf.getVar((Configuration)conf, (MetastoreConf.ConfVars)MetastoreConf.ConfVars.SSL_KEYMANAGERFACTORY_ALGORITHM).trim();
        SslContextFactory.Server factory = new SslContextFactory.Server();
        Object[] excludedProtocols = MetastoreConf.getVar((Configuration)conf, (MetastoreConf.ConfVars)MetastoreConf.ConfVars.SSL_PROTOCOL_BLACKLIST).split(",");
        LOG.info("HTTP Server SSL: adding excluded protocols: " + Arrays.toString(excludedProtocols));
        factory.addExcludeProtocols((String[])excludedProtocols);
        LOG.info("HTTP Server SSL: SslContextFactory.getExcludeProtocols = " + Arrays.toString(factory.getExcludeProtocols()));
        factory.setKeyStorePath(keyStorePath);
        factory.setKeyStorePassword(keyStorePassword);
        factory.setKeyStoreType(keyStoreType);
        factory.setKeyManagerFactoryAlgorithm(keyStoreAlgorithm);
        return factory;
    }

    @FunctionalInterface
    static interface MethodExecutor {
        public void execute(HttpServletRequest var1, HttpServletResponse var2) throws ServletException, IOException;
    }
}

